//	TorusGames2DJigsaw.c
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#include "TorusGames-Common.h"
#include "GeometryGamesUtilities-Common.h"
//#include "GeometryGamesUtilities-Mac-iOS.h"
#include "GeometryGamesSound.h"
#include <math.h>
#include <string.h>	//	for memset()
#ifdef GAME_CONTENT_FOR_SCREENSHOT
#include <stdio.h>	//	for printf()
#endif


#define NUM_2D_BACKGROUND_TEXTURE_REPETITIONS_JIGSAW	7

//	How close must two pieces be to snap together?
#define JIGSAW_EPSILON	0.02

//	When drawing a piece, expand the bounding rectangle as necessary
//	to accommodate either a black outline (on each TabFemale side)
//	or a tab (on each TabMale side).
#define	MARGIN_FOR_BORDER	1.05
#define MARGIN_FOR_TAB		1.53


//	Send a given clump to the front or the back of the draw order?
typedef enum
{
	SendClumpToFront,
	SendClumpToBack
} SendClumpDirection;


//	Public functions with private names
static void	JigsawShutDown(ModelData *md);
static void	JigsawReset(ModelData *md);
static bool	JigsawDragBegin(ModelData *md, bool aRightClick);
static void	JigsawDragObject(ModelData *md, double aHandLocalDeltaH, double aHandLocalDeltaV);
static void	JigsawDragEnd(ModelData *md, double aDragDuration, bool aTouchSequenceWasCancelled);

//	Private functions
static void	SendClumpToFrontOrBack(ModelData *md, unsigned int aPieceIndex, SendClumpDirection aDirection);
static void	MarkClump(JigsawPiece *aPiece);
static void	MoveClump(ModelData *md, JigsawPiece *aPiece, double aDeltaH, double aDeltaV);
static void	SetNeighborPlacements(JigsawPiece *aPiece, TopologyType aTopology);
static bool	BindNearbyNeighbors(ModelData *md);
static void	AlignAllClumps(ModelData *md);
static bool	SetPieceOrientation(JigsawPiece *aPiece, bool aDesiredReflection);
static void	SetNonorientableClump(JigsawPiece *aPiece);
static void	AlignToSymmetryAxes(JigsawPiece *aPiece);
static void	GatherClumps(ModelData *md);
static void	TransferClumpToScratchList(JigsawPiece *aPiece, JigsawPiece **aScratchList, unsigned int *aScratchListIndex);
static bool	PuzzleIsComplete(ModelData *md);


void Jigsaw2DSetUp(ModelData *md)
{
	//	Initialize function pointers.
	md->itsGameShutDown					= &JigsawShutDown;
	md->itsGameReset					= &JigsawReset;
	md->itsGameHumanVsComputerChanged	= NULL;
	md->itsGame2DHandMoved				= NULL;
	md->itsGame2DDragBegin				= &JigsawDragBegin;
	md->itsGame2DDragObject				= &JigsawDragObject;
	md->itsGame2DDragEnd				= &JigsawDragEnd;
	md->itsGame3DDragBegin				= NULL;
	md->itsGame3DDragObject				= NULL;
	md->itsGame3DDragEnd				= NULL;
	md->itsGame3DGridSize				= NULL;
	md->itsGameCharacterInput			= NULL;
	md->itsGameSimulationUpdate			= NULL;
	md->itsGameRefreshMessage			= NULL;
	
	//	These will get set later, in the method
	//
	//		computeJigsawCollageLayout
	//
	//	in the file TorusGamesRenderer.m.
	md->itsGameOf.Jigsaw2D.itsPuzzlePieceCollageFirstCenter	= 0.0;
	md->itsGameOf.Jigsaw2D.itsPuzzlePieceCollageStride		= 0.0;

	//	Start with the last puzzle for each topology,
	//	so the index will wrap around to the first puzzle.
	md->itsGameOf.Jigsaw2D.itsCurrentTorusPuzzle = NUM_TORUS_PUZZLE_IMAGES - 1;
	md->itsGameOf.Jigsaw2D.itsCurrentKleinPuzzle = NUM_KLEIN_PUZZLE_IMAGES - 1;
	
	//	Reset the puzzle.
	JigsawReset(md);
}


static void JigsawShutDown(ModelData *md)
{
}


static void JigsawReset(ModelData *md)
{
	unsigned int	h,
					v,
					n,
					i,
					hh,
					vv;
	JigsawPiece		*thePiece;

#ifdef GAME_CONTENT_FOR_SCREENSHOT
#if 0	//	experiment with different seeds, to see which looks good?
	{
		static unsigned int	theSeed	= 0;

		RandomInitWithSeed(theSeed);
		printf("using 2D maze seed %d\n", theSeed);
		theSeed++;
	}
#else	//	use our favorite seed
	{
		//	Caution:  The random number generator's implementation is different
		//	on different platforms, so while a given seed generates the same jigsaw puzzle
		//	on iOS and macOS, we'd need to use a different seed on other platforms.
		RandomInitWithSeed(2);
	}
#endif	//	experiment or use favorite seed
#endif	//	GAME_CONTENT_FOR_SCREENSHOT

	//	Advance to the next puzzle image.
	if (md->itsTopology == Topology2DTorus)
		md->itsGameOf.Jigsaw2D.itsCurrentTorusPuzzle = (md->itsGameOf.Jigsaw2D.itsCurrentTorusPuzzle + 1) % NUM_TORUS_PUZZLE_IMAGES;
	else
		md->itsGameOf.Jigsaw2D.itsCurrentKleinPuzzle = (md->itsGameOf.Jigsaw2D.itsCurrentKleinPuzzle + 1) % NUM_KLEIN_PUZZLE_IMAGES;
#ifdef GAME_CONTENT_FOR_SCREENSHOT
	md->itsGameOf.Jigsaw2D.itsCurrentTorusPuzzle = 1;	//	the parrot with rich colors
#endif

	//	Choose a puzzle size.
	switch (md->itsDifficultyLevel)
	{
		case 0:  md->itsGameOf.Jigsaw2D.itsSize = 2;  break;
		case 1:  md->itsGameOf.Jigsaw2D.itsSize = 3;  break;
		case 2:  md->itsGameOf.Jigsaw2D.itsSize = 4;  break;
		case 3:  md->itsGameOf.Jigsaw2D.itsSize = 5;  break;
		default: md->itsGameOf.Jigsaw2D.itsSize = 2;  break;	//	should never occur
	}
	if (md->itsGameOf.Jigsaw2D.itsSize > MAX_JIGSAW_SIZE)
		md->itsGameOf.Jigsaw2D.itsSize = MAX_JIGSAW_SIZE;

	//	For easier legibility...
	n = md->itsGameOf.Jigsaw2D.itsSize;

	//	Initialize the pieces.
	for (h = 0; h < n; h++)
	{
		for (v = 0; v < n; v++)
		{
			i = h*n + v;

			md->itsGameOf.Jigsaw2D.itsPieceOrder[i] = &md->itsGameOf.Jigsaw2D.itsPieces[i];

			md->itsGameOf.Jigsaw2D.itsPieces[i].itsLogicalH	= h;
			md->itsGameOf.Jigsaw2D.itsPieces[i].itsLogicalV	= v;

			md->itsGameOf.Jigsaw2D.itsPieces[i].itsPlacement.itsH		= -0.5 + RandomFloat();
			md->itsGameOf.Jigsaw2D.itsPieces[i].itsPlacement.itsV		= -0.5 + RandomFloat();
			md->itsGameOf.Jigsaw2D.itsPieces[i].itsPlacement.itsFlip	= (md->itsTopology == Topology2DKlein ? RandomBoolean() : false);
			md->itsGameOf.Jigsaw2D.itsPieces[i].itsPlacement.itsAngle	= 0.0;
			md->itsGameOf.Jigsaw2D.itsPieces[i].itsPlacement.itsSizeH	= 1.0 / n;
			md->itsGameOf.Jigsaw2D.itsPieces[i].itsPlacement.itsSizeV	= 1.0 / n;

			if (h != 0)
				hh = h - 1;
			else
				hh = n - 1;
			vv = v;
			md->itsGameOf.Jigsaw2D.itsPieces[i].itsNeighborWest		= &md->itsGameOf.Jigsaw2D.itsPieces[hh*n + vv];

			if (h != n - 1)
				hh = h + 1;
			else
				hh = 0;
			vv = v;
			md->itsGameOf.Jigsaw2D.itsPieces[i].itsNeighborEast		= &md->itsGameOf.Jigsaw2D.itsPieces[hh*n + vv];

			hh = h;
			if (v != 0)
			{
				vv = v - 1;
			}
			else
			{
				vv = n - 1;
				if (md->itsTopology == Topology2DKlein)
					hh = (n - 1) - hh;
			}
			md->itsGameOf.Jigsaw2D.itsPieces[i].itsNeighborSouth	= &md->itsGameOf.Jigsaw2D.itsPieces[hh*n + vv];

			hh = h;
			if (v != n - 1)
			{
				vv = v + 1;
			}
			else
			{
				vv = 0;
				if (md->itsTopology == Topology2DKlein)
					hh = (n - 1) - hh;
			}
			md->itsGameOf.Jigsaw2D.itsPieces[i].itsNeighborNorth	= &md->itsGameOf.Jigsaw2D.itsPieces[hh*n + vv];

			md->itsGameOf.Jigsaw2D.itsPieces[i].itsNeighborBondWest		= false;
			md->itsGameOf.Jigsaw2D.itsPieces[i].itsNeighborBondEast		= false;
			md->itsGameOf.Jigsaw2D.itsPieces[i].itsNeighborBondSouth	= false;
			md->itsGameOf.Jigsaw2D.itsPieces[i].itsNeighborBondNorth	= false;

			md->itsGameOf.Jigsaw2D.itsPieces[i].itsClumpIsNonorientable		= false;

			//	Naively one might initialize itsTextureBordersNeedUpdate to true
			//	to let the puzzle's "collage" get one full update at the first opportunity.
			//	But in practice that isn't really necessary, because the collage starts
			//	with all borders present, and until the user joins two pieces together,
			//	no borders need to be erased.  So we may safely initialize
			//	itsTextureBordersNeedUpdate to false.  On my iPad Air 2,
			//	this saves about 0.1 sec, when using 2048×2048 puzzle images.
			//
			md->itsGameOf.Jigsaw2D.itsPieces[i].itsTextureBordersNeedUpdate	= false;
		}
	}

	//	Set the tab genders randomly.
	for (i = 0; i < md->itsGameOf.Jigsaw2D.itsSize * md->itsGameOf.Jigsaw2D.itsSize; i++)
	{
		thePiece = &md->itsGameOf.Jigsaw2D.itsPieces[i];

#ifdef MAKE_GAME_CHOICE_ICONS
		//	For the game choice icon, make all west and north sides male, and all east and south female.

		thePiece->itsTabGenderWest  = TabMale;
		thePiece->itsNeighborWest->itsTabGenderEast   = OPPOSITE_GENDER(thePiece->itsTabGenderWest);

		thePiece->itsTabGenderSouth = TabFemale;
		thePiece->itsNeighborSouth->itsTabGenderNorth = OPPOSITE_GENDER(thePiece->itsTabGenderSouth);
#else
		thePiece->itsTabGenderWest  = (RandomBoolean() ? TabFemale : TabMale);
		thePiece->itsNeighborWest->itsTabGenderEast   = OPPOSITE_GENDER(thePiece->itsTabGenderWest);

		thePiece->itsTabGenderSouth = (RandomBoolean() ? TabFemale : TabMale);
		thePiece->itsNeighborSouth->itsTabGenderNorth = OPPOSITE_GENDER(thePiece->itsTabGenderSouth);
#endif
	}

	//	Note:  We rely on the root controller to call
	//	a method like -resetGameForPurpose: that triggers
	//	a fresh call to -setUpTexturesWithModelData: to rebuild
	//
	//		itsTextures[Texture2DJigsawCollageWithoutBorders]
	//		itsTextures[Texture2DJigsawCollageWithBorders]
	//

	//	Abort any pending simulation.
	SimulationEnd(md);	//	unused in Jigsaw2D

	//	Ready to go.
	md->itsGameIsOver = false;
}


static bool JigsawDragBegin(
	ModelData	*md,
	bool		aRightClick)
{
	unsigned int	n;
	double			thePieceHalfSize;
	unsigned int	i;
	JigsawPiece		*thePiece;
	Placement2D		thePieceGlobal,
					theHandGlobal,
					theHandLocal;
	TabGender		theTabGenderH,
					theTabGenderV;
	unsigned int	h,
					v,
					theTexelCoordsH[2],
					theTexelCoordsV[2];
	char			theASCIITexelH,
					theASCIITexelV;
	bool			theOpacityH,
					theOpacityV;

	UNUSED_PARAMETER(aRightClick);

#define ASCII_TEMPLATE_WIDTH	64
#define ASCII_TEMPLATE_HEIGHT	32
	static const char	theASCIIPieceTemplate[ASCII_TEMPLATE_HEIGHT][ASCII_TEMPLATE_WIDTH] =
						{
							"+++++++++++++++++++++++++++++++++++++++++++++++**---------------",
							"+++++++++++++++++++++++++++++++++++++++++++++++**---------------",
							"+++++++++++++++++++++++++++++++++++++++++++++++**---------------",
							"++++++++++++++++++++++++++++++++++++**+++++++++**---------------",
							"+++++++++++++++++++++++++++++++++++****++++++++**---------------",
							"++++++++++++++++++++++++++++++++++*----*+++++++*----------------",
							"+++++++++++++++++++++++++++++++++**----**++++++*----------------",
							"+++++++++++++++++++++++++++++++++*------*++++++*----------------",
							"++++++++++++++++++++++++++++++++**------**++++*-----------------",
							"++++++++++++++++++++++++++++++++**-------**++**-----------------",
							"++++++++++++++++++++++++++++++++*---------****------------------",
							"++++++++++++++++++++++++++++++++*-------------------------------",
							"++++++++++++++++++++++++++++++++*-------------------------------",
							"++++++++++++++++++++++++++++++++*-------------------------------",
							"++++++++++++++++++++++++++++++++*-------------------------------",
							"++++++++++++++++++++++++++++++++*-------------------------------",
							"+++++++++++++++++++++++++++++++**-------------------------------",
							"+++++++++++++++++++++++++++++++**-------------------------------",
							"+++++++++++++++++++++++++++++++**-------------------------------",
							"+++++++++++++++++++++++++++++++**-------------------------------",
							"+++++++++++++++++++++++++++++++**-------------------------------",
							"+++++++++++++++++++++++++++++++**-------------------------------",
							"+++++++++++++++++++++++++++++++**-------------------------------",
							"+++++++++++++++++++++++++++++++**-------------------------------",
							"+++++++++++++++++++++++++++++++**-------------------------------",
							"+++++++++++++++++++++++++++++++**-------------------------------",
							"+++++++++++++++++++++++++++++++**-------------------------------",
							"+++++++++++++++++++++++++++++++**-------------------------------",
							"+++++++++++++++++++++++++++++++**-------------------------------",
							"+++++++++++++++++++++++++++++++**-------------------------------",
							"+++++++++++++++++++++++++++++++**-------------------------------",
							"+++++++++++++++++++++++++++++++**-------------------------------"
						};

	//	If the game is already over, ignore all hits.
	if (md->itsGameIsOver)
		return false;

	//	For brevity
	n = md->itsGameOf.Jigsaw2D.itsSize;
	
	//	Note the piece size, halved for convenience.
	thePieceHalfSize = 0.5 / n;

	//	Search the array md->itsGameOf.Jigsaw2D.itsPieceOrder[]
	//	in forwards order for top-to-bottom hit testing.
	for (i = 0; i < n*n; i++)
	{
		thePiece		= md->itsGameOf.Jigsaw2D.itsPieceOrder[i];
		thePieceGlobal	= thePiece->itsPlacement;
		theHandGlobal	= md->its2DHandPlacement;

		//	Locate the hand image nearest to the piece.
		if (theHandGlobal.itsV - thePieceGlobal.itsV < -0.5)
		{
			theHandGlobal.itsV += 1.0;
			if (md->itsTopology == Topology2DKlein)
			{
				theHandGlobal.itsH		= -theHandGlobal.itsH;
				theHandGlobal.itsFlip	= ! theHandGlobal.itsFlip;
			}
		}
		if (theHandGlobal.itsV - thePieceGlobal.itsV > +0.5)
		{
			theHandGlobal.itsV -= 1.0;
			if (md->itsTopology == Topology2DKlein)
			{
				theHandGlobal.itsH		= -theHandGlobal.itsH;
				theHandGlobal.itsFlip	= ! theHandGlobal.itsFlip;
			}
		}
		if (theHandGlobal.itsH - thePieceGlobal.itsH < -0.5)
			theHandGlobal.itsH += 1.0;
		if (theHandGlobal.itsH - thePieceGlobal.itsH > +0.5)
			theHandGlobal.itsH -= 1.0;

		//	Get the hand location in the piece's (unflipped) coordinate system,
		//	keeping in mind that in a Klein bottle puzzle,
		//	the piece itself might be flipped.
		//
		theHandLocal.itsH = theHandGlobal.itsH - thePieceGlobal.itsH;
		theHandLocal.itsV = theHandGlobal.itsV - thePieceGlobal.itsV;
		if ( ! thePieceGlobal.itsFlip )
		{
			theHandLocal.itsFlip	= theHandGlobal.itsFlip;
		}
		else
		{
			theHandLocal.itsFlip	= ! theHandGlobal.itsFlip;
			theHandLocal.itsH		= - theHandLocal.itsH;
		}
		theHandLocal.itsAngle	= 0.0;	//	unused
		theHandLocal.itsSizeH	= 0.0;	//	unused
		theHandLocal.itsSizeV	= 0.0;	//	unused
		
		//	If the hand is near the piece, test for a hit.
		//
		if (fabs(theHandLocal.itsH) < thePieceHalfSize * MARGIN_FOR_TAB
		 && fabs(theHandLocal.itsV) < thePieceHalfSize * MARGIN_FOR_TAB)
		{
			//	Note whether the nearby sides of the puzzle piece
			//	have slots or tabs.
			theTabGenderH = (theHandLocal.itsH >= 0.0 ?
								thePiece->itsTabGenderEast  :
								thePiece->itsTabGenderWest    );
			theTabGenderV = (theHandLocal.itsV >= 0.0 ?
								thePiece->itsTabGenderNorth :
								thePiece->itsTabGenderSouth   );
			
			//	Let
			//
			//		h be the pixel's non-negative horizontal coordinate,
			//			which may run leftward or rightward,
			//			in ASCII array coordinates
			//	and
			//		v be the pixel's non-negative vertical coordinate,
			//			which may run downward or upward,
			//			in ASCII array coordinates.
			//
			//	Note:
			//		We multiply by ASCII_TEMPLATE_WIDTH
			//		in both the horizontal and vertical directions.
			//		In effect we're pretending that the ASCII template
			//		is twice as tall as it really is.
			//		We'll clamp the vertical texture coordinate
			//		to stay within the ASCII array's bounds.
			//
			h = (unsigned int) (fabs(theHandLocal.itsH) * n * ASCII_TEMPLATE_WIDTH);
			v = (unsigned int) (fabs(theHandLocal.itsV) * n * ASCII_TEMPLATE_WIDTH);	//	not a typo!

			//	Locate the ASCII texel for the shape of the piece's left or right side.
			theTexelCoordsH[0] = MIN(h, ASCII_TEMPLATE_WIDTH  - 1);  //	marginal clamping
			theTexelCoordsH[1] = MIN(v, ASCII_TEMPLATE_HEIGHT - 1);	//	 lots of clamping
			if (theTabGenderH == TabFemale)
				theTexelCoordsH[0] = (ASCII_TEMPLATE_WIDTH - 1) - theTexelCoordsH[0];
			theASCIITexelH = theASCIIPieceTemplate[theTexelCoordsH[1]][theTexelCoordsH[0]];
			theOpacityH = (	theTabGenderH == TabFemale ?
							(theASCIITexelH == '-' || theASCIITexelH == '*') :
							(theASCIITexelH == '+' || theASCIITexelH == '*') );

			//	Swap the roles of h and v to locate the ASCII texel
			//	for the shape of the piece's bottom or top side.
			theTexelCoordsV[0] = MIN(v, ASCII_TEMPLATE_WIDTH  - 1);  //	marginal clamping
			theTexelCoordsV[1] = MIN(h, ASCII_TEMPLATE_HEIGHT - 1);	//	 lots of clamping
			if (theTabGenderV == TabFemale)
				theTexelCoordsV[0] = (ASCII_TEMPLATE_WIDTH - 1) - theTexelCoordsV[0];
			theASCIITexelV = theASCIIPieceTemplate[theTexelCoordsV[1]][theTexelCoordsV[0]];
			theOpacityV = (	theTabGenderV == TabFemale ?
							(theASCIITexelV == '-' || theASCIITexelV == '*') :
							(theASCIITexelV == '+' || theASCIITexelV == '*') );
	
			//	Did we hit an opaque texel?
			if (theOpacityH && theOpacityV)
			{
				md->its2DDragIsReversed				= theHandLocal.itsFlip;
				md->itsGameOf.Jigsaw2D.itsDragPiece	= thePiece;
				SendClumpToFrontOrBack(md, i, SendClumpToFront);

				return true;
			}
		}
	}

	return false;
}


static void SendClumpToFrontOrBack(
	ModelData			*md,
	unsigned int		aPieceIndex,		//	an arbitrary piece within the clump
	SendClumpDirection	aDirection)
{
	unsigned int	thePuzzleSize,
					theNumPieces;
	JigsawPiece		*theTempList[MAX_JIGSAW_SIZE * MAX_JIGSAW_SIZE];
	unsigned int	i,
					theWriteIndex;
	bool			theFrontFlag,
					theBackFlag;
	
	thePuzzleSize	= md->itsGameOf.Jigsaw2D.itsSize;
	theNumPieces	= thePuzzleSize * thePuzzleSize;
	GEOMETRY_GAMES_ASSERT(theNumPieces > 0, "Jigsaw puzzle has zero size");

	//	"Unnecessary" error check.
	if (aPieceIndex >= theNumPieces)
	{
		GeometryGamesErrorMessage(u"Bad piece index", u"Error in SendClumpToFrontOrBack()");
		return;
	}

	//	Set itsVisitedFlag to false for all pieces.
	for (i = 0; i < theNumPieces; i++)
		md->itsGameOf.Jigsaw2D.itsPieceOrder[i]->itsVisitedFlag = false;

	//	Visit and mark all pieces in the given clump.
	MarkClump(md->itsGameOf.Jigsaw2D.itsPieceOrder[aPieceIndex]);

	//	Initialize a "write index" on theTempList.
	theWriteIndex = 0;

	//	Which pieces (marked or unmarked) should move to the front of the list?
	theFrontFlag	= (aDirection == SendClumpToFront);
	theBackFlag		= ! theFrontFlag;

	//	Copy pointers to front-bound pieces onto theTempList.
	for (i = 0; i < theNumPieces; i++)
		if (md->itsGameOf.Jigsaw2D.itsPieceOrder[i]->itsVisitedFlag == theFrontFlag)
			theTempList[theWriteIndex++] = md->itsGameOf.Jigsaw2D.itsPieceOrder[i];

	//	Copy pointers to back-bound pieces onto theTempList,
	//	following the front-bound pieces.
	for (i = 0; i < theNumPieces; i++)
		if (md->itsGameOf.Jigsaw2D.itsPieceOrder[i]->itsVisitedFlag == theBackFlag)
			theTempList[theWriteIndex++] = md->itsGameOf.Jigsaw2D.itsPieceOrder[i];

	//	"Unnecessary" error check.
	if (theWriteIndex != theNumPieces)
	{
		GeometryGamesErrorMessage(u"Internal error while copying pieces.", u"Error in SendClumpToFrontOrBack()");
		return;
	}

	//	Copy theTempList back to itsGameOf.Jigsaw2D.itsPieceOrder.
	for (i = 0; i < theNumPieces; i++)
		md->itsGameOf.Jigsaw2D.itsPieceOrder[i] = theTempList[i];
}


static void MarkClump(JigsawPiece *aPiece)
{
	if ( ! aPiece->itsVisitedFlag )
	{
		aPiece->itsVisitedFlag = true;

		if (aPiece->itsNeighborBondWest)	MarkClump(aPiece->itsNeighborWest);
		if (aPiece->itsNeighborBondEast)	MarkClump(aPiece->itsNeighborEast);
		if (aPiece->itsNeighborBondSouth)	MarkClump(aPiece->itsNeighborSouth);
		if (aPiece->itsNeighborBondNorth)	MarkClump(aPiece->itsNeighborNorth);
	}
}


static void JigsawDragObject(
	ModelData	*md,
	double		aHandLocalDeltaH,
	double		aHandLocalDeltaV)
{
	double	thePieceLocalDeltaH,
			thePieceLocalDeltaV,
			theHandAmbientDeltaH,
			theHandAmbientDeltaV,
			thePieceAmbientDeltaH,
			thePieceAmbientDeltaV;

	//	Drag the clump containing itsDragPiece,
	//	which should be the topmost clump.

	thePieceLocalDeltaH		= md->its2DDragIsReversed
								? -aHandLocalDeltaH : aHandLocalDeltaH;
	thePieceLocalDeltaV		= aHandLocalDeltaV;

	theHandAmbientDeltaH	= md->its2DHandPlacement.itsFlip
								? -aHandLocalDeltaH : aHandLocalDeltaH;
	theHandAmbientDeltaV	= aHandLocalDeltaV;

	thePieceAmbientDeltaH	= md->itsGameOf.Jigsaw2D.itsDragPiece->itsPlacement.itsFlip
								? -thePieceLocalDeltaH : thePieceLocalDeltaH;
	thePieceAmbientDeltaV	= thePieceLocalDeltaV;

	//	Move the hand.
	md->its2DHandPlacement.itsH += theHandAmbientDeltaH;
	md->its2DHandPlacement.itsV += theHandAmbientDeltaV;
	Normalize2DPlacement(&md->its2DHandPlacement, md->itsTopology);

	//	In a Klein bottle a clump of pieces may reach all the way
	//	around and connect up with itself in such a way that the clump
	//	cannot move left or right.
	if (md->itsGameOf.Jigsaw2D.itsDragPiece->itsClumpIsNonorientable)
		thePieceAmbientDeltaH = 0.0;

	//	Move the clump of pieces containing itsDragPiece.
	MoveClump(	md,
				md->itsGameOf.Jigsaw2D.itsDragPiece,
				thePieceAmbientDeltaH,
				thePieceAmbientDeltaV);
}


static void MoveClump(
	ModelData	*md,
	JigsawPiece	*aPiece,
	double		aDeltaH,
	double		aDeltaV)
{
	unsigned int	i;

	//	Move aPiece.
	aPiece->itsPlacement.itsH += aDeltaH;
	aPiece->itsPlacement.itsV += aDeltaV;

	//	Move all pieces in the same clump.

	//	The recursive function SetNeighborPlacements() gets the placements
	//	right in the universal cover (using itsVisitedFlag to avoid visiting
	//	the same piece twice), then after it's done Normalize2DPlacement()
	//	projects each piece individually to the fundamental domain.

	for (i = md->itsGameOf.Jigsaw2D.itsSize * md->itsGameOf.Jigsaw2D.itsSize; i-- > 0; )
		md->itsGameOf.Jigsaw2D.itsPieces[i].itsVisitedFlag = false;

	SetNeighborPlacements(aPiece, md->itsTopology);

	for (i = md->itsGameOf.Jigsaw2D.itsSize * md->itsGameOf.Jigsaw2D.itsSize; i-- > 0; )	//	just do 'em all -- it's fast
		Normalize2DPlacement(&md->itsGameOf.Jigsaw2D.itsPieces[i].itsPlacement, md->itsTopology);
}


static void SetNeighborPlacements(
	JigsawPiece		*aPiece,
	TopologyType	aTopology)
{
	//	Assume aPiece->itsPlacement has already been set
	//	and transfer the placement to all connected neighbors,
	//	and so on recursively.
	//	Use itsVisitedFlag to avoid setting the same piece twice.

	aPiece->itsVisitedFlag = true;

	if (aPiece->itsNeighborBondWest
	 && ! aPiece->itsNeighborWest->itsVisitedFlag)
	{
		aPiece->itsNeighborWest->itsPlacement = aPiece->itsPlacement;

		if (aPiece->itsPlacement.itsFlip)
			aPiece->itsNeighborWest->itsPlacement.itsH += aPiece->itsPlacement.itsSizeH;
		else
			aPiece->itsNeighborWest->itsPlacement.itsH -= aPiece->itsPlacement.itsSizeH;

		SetNeighborPlacements(aPiece->itsNeighborWest, aTopology);
	}

	if (aPiece->itsNeighborBondEast
	 && ! aPiece->itsNeighborEast->itsVisitedFlag)
	{
		aPiece->itsNeighborEast->itsPlacement = aPiece->itsPlacement;

		if (aPiece->itsPlacement.itsFlip)
			aPiece->itsNeighborEast->itsPlacement.itsH -= aPiece->itsPlacement.itsSizeH;
		else
			aPiece->itsNeighborEast->itsPlacement.itsH += aPiece->itsPlacement.itsSizeH;

		SetNeighborPlacements(aPiece->itsNeighborEast, aTopology);
	}

	if (aPiece->itsNeighborBondSouth
	 && ! aPiece->itsNeighborSouth->itsVisitedFlag)
	{
		aPiece->itsNeighborSouth->itsPlacement = aPiece->itsPlacement;

		aPiece->itsNeighborSouth->itsPlacement.itsV -= aPiece->itsPlacement.itsSizeV;

		aPiece->itsNeighborSouth->itsPlacement.itsFlip
			^= ( aTopology == Topology2DKlein  &&  aPiece->itsLogicalV == 0 );

		SetNeighborPlacements(aPiece->itsNeighborSouth, aTopology);
	}

	if (aPiece->itsNeighborBondNorth
	 && ! aPiece->itsNeighborNorth->itsVisitedFlag)
	{
		aPiece->itsNeighborNorth->itsPlacement = aPiece->itsPlacement;

		aPiece->itsNeighborNorth->itsPlacement.itsV += aPiece->itsPlacement.itsSizeV;

		aPiece->itsNeighborNorth->itsPlacement.itsFlip
			^= ( aTopology == Topology2DKlein  &&  aPiece->itsNeighborNorth->itsLogicalV == 0 );

		SetNeighborPlacements(aPiece->itsNeighborNorth, aTopology);
	}
}


static void JigsawDragEnd(
	ModelData	*md,
	double		aDragDuration,	//	in seconds
	bool		aTouchSequenceWasCancelled)
{
	//	Nudging a piece B into position next to a nearby piece C
	//	might in turn align piece B with some previously tested piece A.
	//	To allow for this "ripple effect" let's keep repeating the alignment
	//	process until no further progress occurs.  Typically, of course,
	//	one pass will suffice.

	bool	theSomePiecesGotJoinedFlag;

	UNUSED_PARAMETER(aDragDuration);

	//	If a touch sequence got cancelled (perhaps because a gesture was recognized),
	//	return without binding nearby neighbors or testing for a win.
	if (aTouchSequenceWasCancelled)
		return;

	theSomePiecesGotJoinedFlag = false;

	while (BindNearbyNeighbors(md))
	{
		//	Align the pieces within each clump.
		//	Detect and label nonorientable clumps,
		//	and nudge them to align with the symmetry axes.
		AlignAllClumps(md);

		//	Note that something happened.
		theSomePiecesGotJoinedFlag = true;
	}

	if (theSomePiecesGotJoinedFlag)
	{
		GatherClumps(md);
		EnqueueSoundRequest(u"JigsawJoin.mid");
	}

	if (PuzzleIsComplete(md))
	{
		md->itsGameIsOver = true;
		EnqueueSoundRequest(u"JigsawComplete.mid");
	}
}


static bool BindNearbyNeighbors(ModelData *md)
{
	bool			theProgressFlag,
					theFlipsAreConsistent;
	unsigned int	i;
	JigsawPiece		*thePiece;

	theProgressFlag = false;

	for (i = md->itsGameOf.Jigsaw2D.itsSize * md->itsGameOf.Jigsaw2D.itsSize; i-- > 0; )
	{
		thePiece = &md->itsGameOf.Jigsaw2D.itsPieces[i];

		theFlipsAreConsistent = (thePiece->itsPlacement.itsFlip == thePiece->itsNeighborWest->itsPlacement.itsFlip);
		if
		(	! thePiece->itsNeighborBondWest
		 && Shortest2DDistance(
				thePiece->itsPlacement.itsFlip ?
					thePiece->itsPlacement.itsH + thePiece->itsPlacement.itsSizeH :
					thePiece->itsPlacement.itsH - thePiece->itsPlacement.itsSizeV,
				thePiece->itsPlacement.itsV,
				thePiece->itsNeighborWest->itsPlacement.itsH,
				thePiece->itsNeighborWest->itsPlacement.itsV,
				md->itsTopology,
				&theFlipsAreConsistent) < JIGSAW_EPSILON
		 && theFlipsAreConsistent )
		{
			//	Strictly speaking it's redundant to "set the bond"
			//	in both directions (because the loop will eventually
			//	visit both thePiece and thePiece->itsNeighborWest)
			//	but setting both directions at once guarantees
			//	"bond symmetry" without depending on consistent
			//	floating point results.
			thePiece->itsNeighborBondWest							= true;
			thePiece->itsTextureBordersNeedUpdate					= true;
			thePiece->itsNeighborWest->itsNeighborBondEast			= true;
			thePiece->itsNeighborWest->itsTextureBordersNeedUpdate	= true;

			theProgressFlag = true;
		}

		theFlipsAreConsistent = (thePiece->itsPlacement.itsFlip == thePiece->itsNeighborEast->itsPlacement.itsFlip);
		if
		(	! thePiece->itsNeighborBondEast
		 && Shortest2DDistance(
				thePiece->itsPlacement.itsFlip ?
					thePiece->itsPlacement.itsH - thePiece->itsPlacement.itsSizeH :
					thePiece->itsPlacement.itsH + thePiece->itsPlacement.itsSizeH,
				thePiece->itsPlacement.itsV,
				thePiece->itsNeighborEast->itsPlacement.itsH,
				thePiece->itsNeighborEast->itsPlacement.itsV,
				md->itsTopology,
				&theFlipsAreConsistent) < JIGSAW_EPSILON
		 && theFlipsAreConsistent )
		{
			thePiece->itsNeighborBondEast							= true;
			thePiece->itsTextureBordersNeedUpdate					= true;
			thePiece->itsNeighborEast->itsNeighborBondWest			= true;
			thePiece->itsNeighborEast->itsTextureBordersNeedUpdate	= true;

			theProgressFlag = true;
		}

		theFlipsAreConsistent = (
			(thePiece->itsPlacement.itsFlip == thePiece->itsNeighborSouth->itsPlacement.itsFlip)
		  ^ (md->itsTopology == Topology2DKlein && thePiece->itsLogicalV == 0) );
		if
		(	! thePiece->itsNeighborBondSouth
		 && Shortest2DDistance(
				thePiece->itsPlacement.itsH,
				thePiece->itsPlacement.itsV - thePiece->itsPlacement.itsSizeV,
				thePiece->itsNeighborSouth->itsPlacement.itsH,
				thePiece->itsNeighborSouth->itsPlacement.itsV,
				md->itsTopology,
				&theFlipsAreConsistent) < JIGSAW_EPSILON
		 && theFlipsAreConsistent )
		{
			thePiece->itsNeighborBondSouth							= true;
			thePiece->itsTextureBordersNeedUpdate					= true;
			thePiece->itsNeighborSouth->itsNeighborBondNorth		= true;
			thePiece->itsNeighborSouth->itsTextureBordersNeedUpdate	= true;

			theProgressFlag = true;
		}

		theFlipsAreConsistent = (
			(thePiece->itsPlacement.itsFlip == thePiece->itsNeighborNorth->itsPlacement.itsFlip)
		  ^ (md->itsTopology == Topology2DKlein && thePiece->itsNeighborNorth->itsLogicalV == 0) );
		if
		(
			! thePiece->itsNeighborBondNorth
		 && Shortest2DDistance(
				thePiece->itsPlacement.itsH,
				thePiece->itsPlacement.itsV + thePiece->itsPlacement.itsSizeV,
				thePiece->itsNeighborNorth->itsPlacement.itsH,
				thePiece->itsNeighborNorth->itsPlacement.itsV,
				md->itsTopology,
				&theFlipsAreConsistent) < JIGSAW_EPSILON
		 && theFlipsAreConsistent )
		{
			thePiece->itsNeighborBondNorth							= true;
			thePiece->itsTextureBordersNeedUpdate					= true;
			thePiece->itsNeighborNorth->itsNeighborBondSouth		= true;
			thePiece->itsNeighborNorth->itsTextureBordersNeedUpdate	= true;

			theProgressFlag = true;
		}
	}

	return theProgressFlag;
}


static void AlignAllClumps(ModelData *md)
{
	unsigned int	i;

	//	Detect nonorientable clumps.

	if (md->itsTopology == Topology2DKlein)
	{
		//	Initialize itsVisitedFlag to be false and
		//	(tentatively!) assume all clumps to be orientable.
		for (i = md->itsGameOf.Jigsaw2D.itsSize * md->itsGameOf.Jigsaw2D.itsSize; i-- > 0; )
		{
			md->itsGameOf.Jigsaw2D.itsPieces[i].itsVisitedFlag			= false;
			md->itsGameOf.Jigsaw2D.itsPieces[i].itsClumpIsNonorientable	= false;
		}

		//	For each...
		for (i = md->itsGameOf.Jigsaw2D.itsSize * md->itsGameOf.Jigsaw2D.itsSize; i-- > 0; )
		{
			//	...unvisited piece...
			if ( ! md->itsGameOf.Jigsaw2D.itsPieces[i].itsVisitedFlag )
			{
				//	...imagine that piece's clump sitting in the universal cover.
				//	Set itsReflectionFlag = false (resp. true) for each piece
				//	in the clump whose orientation agrees (resp. disagrees)
				//	with the initial piece's orientation, and set itsVisitedFlag = true
				//	as well.  If the clump connects up with itself
				//	in an orientation-reversing way...
				if ( ! SetPieceOrientation(&md->itsGameOf.Jigsaw2D.itsPieces[i], false) )
				{
					//	...then align each piece in the clump to the symmetry axes
					//	and set itsClumpIsNonorientable to true.
					SetNonorientableClump(&md->itsGameOf.Jigsaw2D.itsPieces[i]);
				}
			}
		}
	}

	//	Align pieces within each clump.

	//		Initialize itsVisitedFlag to false;
	for (i = md->itsGameOf.Jigsaw2D.itsSize * md->itsGameOf.Jigsaw2D.itsSize; i-- > 0; )
		md->itsGameOf.Jigsaw2D.itsPieces[i].itsVisitedFlag = false;

	//		Align the pieces in the universal cover,
	//		setting itsVisitedFlag as we go along.
	for (i = md->itsGameOf.Jigsaw2D.itsSize * md->itsGameOf.Jigsaw2D.itsSize; i-- > 0; )
		if ( ! md->itsGameOf.Jigsaw2D.itsPieces[i].itsVisitedFlag )
			SetNeighborPlacements(&md->itsGameOf.Jigsaw2D.itsPieces[i], md->itsTopology);

	//		Project each piece to the fundamental domain.
	for (i = md->itsGameOf.Jigsaw2D.itsSize * md->itsGameOf.Jigsaw2D.itsSize; i-- > 0; )
		Normalize2DPlacement(&md->itsGameOf.Jigsaw2D.itsPieces[i].itsPlacement, md->itsTopology);
}


static bool SetPieceOrientation(
	JigsawPiece	*aPiece,
	bool		aDesiredReflection)
{
	if ( ! aPiece->itsVisitedFlag )
	{
		//	aPiece has not yet been visited.

		//	Assign aDesiredReflection to this piece.
		aPiece->itsVisitedFlag		= true;
		aPiece->itsReflectionFlag	= aDesiredReflection;

		//	Visit its neighbors.
		//	Return false if inconsistencies arise.
		if (aPiece->itsNeighborBondWest)
		{
			if ( ! SetPieceOrientation(aPiece->itsNeighborWest, aDesiredReflection) )
				return false;
		}
		if (aPiece->itsNeighborBondEast)
		{
			if ( ! SetPieceOrientation(aPiece->itsNeighborEast, aDesiredReflection) )
				return false;
		}
		if (aPiece->itsNeighborBondSouth)
		{
			if ( ! SetPieceOrientation(
					aPiece->itsNeighborSouth,
					aPiece->itsLogicalV == 0 ? !aDesiredReflection : aDesiredReflection) )
				return false;
		}
		if (aPiece->itsNeighborBondNorth)
		{
			if ( ! SetPieceOrientation(
					aPiece->itsNeighborNorth,
					aPiece->itsNeighborNorth->itsLogicalV == 0 ? !aDesiredReflection : aDesiredReflection) )
				return false;
		}

		//	No inconsistencies arose, so return true.
		return true;
	}
	else
	{
		//	aPiece has already been visited.
		return (aPiece->itsReflectionFlag == aDesiredReflection);
	}

}


static void SetNonorientableClump(JigsawPiece *aPiece)
{
	if ( ! aPiece->itsClumpIsNonorientable )
	{
		aPiece->itsClumpIsNonorientable = true;

		AlignToSymmetryAxes(aPiece);

		if (aPiece->itsNeighborBondWest)
			SetNonorientableClump(aPiece->itsNeighborWest);
		if (aPiece->itsNeighborBondEast)
			SetNonorientableClump(aPiece->itsNeighborEast);
		if (aPiece->itsNeighborBondSouth)
			SetNonorientableClump(aPiece->itsNeighborSouth);
		if (aPiece->itsNeighborBondNorth)
			SetNonorientableClump(aPiece->itsNeighborNorth);
	}
}


static void AlignToSymmetryAxes(JigsawPiece *aPiece)
{
	//	If a piece sits in a nonorientable clump, then its logical
	//	coordinates determine its distance from the symmetry axes.
	//	The Klein bottle has two symmetry axes (at x = 0 and x = -0.5 = +0.5)
	//	so a piece's position is well-defined only up to multiples of 0.5.
	//	Moreover, when a piece is flipped it must lie on the
	//	"opposite side" of the given symmetry axis.

	double	thePredictedH,
			theError;

	//	Compute the canonical horizontal coordinate of aPiece's center.
	thePredictedH = -0.5 + (0.5 + aPiece->itsLogicalH)*(aPiece->itsPlacement.itsSizeH);

	//	If aPiece is flipped, it lies on the opposite side of the symmetry axis.
	//	(This is true for both symmetry axes simultaneously!)
	if (aPiece->itsPlacement.itsFlip)
		thePredictedH = - thePredictedH;

	//	There are two ways to assemble the jigsaw puzzle:
	//	one way preserves the symmetry axes individually while
	//	the other way interchanges them.  A horizontal translation
	//	of 1/2 (mod 1) takes one way to the other.
	theError = fabs(aPiece->itsPlacement.itsH - thePredictedH);
	if (theError > 0.5)		//	account for wraparound
		theError = 1.0 - theError;
	if (theError > 0.25)	//	switch axes if necessary
	{
		if (thePredictedH < 0.0)
			thePredictedH += 0.5;
		else
			thePredictedH -= 0.5;
	}
	aPiece->itsPlacement.itsH = thePredictedH;
}


static void GatherClumps(ModelData *md)
{
	//	Technical note:  It would almost always work fine
	//	to gather only the clump containing itsDragPiece,
	//	but for robustness let's gather all clumps (so for example
	//	if two background pieces happen to sit within epsilon
	//	of their correct relative position just by accident, and
	//	get attached with no user initiative, this algorithm
	//	will bring them together on the list).

	unsigned int	thePuzzleSize,
					theNumPieces;
	unsigned int	i,	//	index on main draw order list
					ii;	//	index on scratch list
	JigsawPiece		*theScratchList[MAX_JIGSAW_SIZE * MAX_JIGSAW_SIZE];

	thePuzzleSize	= md->itsGameOf.Jigsaw2D.itsSize;
	theNumPieces	= thePuzzleSize * thePuzzleSize;
	GEOMETRY_GAMES_ASSERT(theNumPieces > 0, "Jigsaw puzzle has zero size");

	//	Initialize itsVisitedFlag to false;
	for (i = 0; i < theNumPieces; i++)
		md->itsGameOf.Jigsaw2D.itsPieces[i].itsVisitedFlag = false;

	//	Go down the permanent list and copy pieces to theScratchList
	//	on a clump by clump basis.
	//	Use a recursion to "sniff out" each clump,
	//	marking pieces as visited as we copy them.
	for (i = 0, ii = 0; i < theNumPieces; i++)
		TransferClumpToScratchList(md->itsGameOf.Jigsaw2D.itsPieceOrder[i], theScratchList, &ii);

	//	An "unnecessary" error check.
	if (ii != theNumPieces)
	{
		GeometryGamesErrorMessage(u"Wrong number of pieces transferred to theScratchList", u"Error in GatherClumps()");
		return;
	}

	//	Copy theScratchList back to the permanent list.
	for (i = 0; i < theNumPieces; i++)
		md->itsGameOf.Jigsaw2D.itsPieceOrder[i] = theScratchList[i];
}


static void TransferClumpToScratchList(
	JigsawPiece		*aPiece,
	JigsawPiece		**aScratchList,
	unsigned int	*aScratchListIndex)
{
	if ( ! aPiece->itsVisitedFlag )
	{
		aPiece->itsVisitedFlag = true;

		aScratchList[(*aScratchListIndex)++] = aPiece;

		if (aPiece->itsNeighborBondWest)
			TransferClumpToScratchList(aPiece->itsNeighborWest,  aScratchList, aScratchListIndex);
		if (aPiece->itsNeighborBondEast)
			TransferClumpToScratchList(aPiece->itsNeighborEast,  aScratchList, aScratchListIndex);
		if (aPiece->itsNeighborBondSouth)
			TransferClumpToScratchList(aPiece->itsNeighborSouth, aScratchList, aScratchListIndex);
		if (aPiece->itsNeighborBondNorth)
			TransferClumpToScratchList(aPiece->itsNeighborNorth, aScratchList, aScratchListIndex);
	}
}


static bool PuzzleIsComplete(ModelData *md)
{
	unsigned int	i;

	for (i = md->itsGameOf.Jigsaw2D.itsSize * md->itsGameOf.Jigsaw2D.itsSize; i-- > 0; )
	{
		if (! md->itsGameOf.Jigsaw2D.itsPieces[i].itsNeighborBondWest
		 || ! md->itsGameOf.Jigsaw2D.itsPieces[i].itsNeighborBondEast
		 || ! md->itsGameOf.Jigsaw2D.itsPieces[i].itsNeighborBondSouth
		 || ! md->itsGameOf.Jigsaw2D.itsPieces[i].itsNeighborBondNorth )
		{
			return false;
		}
	}

	return true;
}


//	Torus Games once had a Send to Back menu item
//	to send the topmost clump to the back of the stacking order.
//	It was occasionally useful in a Klein bottle puzzle
//	when a multiconnected clump of pieces would "lock up"
//	(meaning that it could no longer be slid leftward or rightward)
//	thus blocking access to other pieces underneath it.
//	However, the feature was rarely needed and even more rarely used,
//	so I removed it to simplify the menus.
//	If I ever want that feature back, 
//	here's the function that implements it:
//
//	static void JigsawSpecialCommand(ModelData *md)
//	{
//		//	Send the frontmost clump (the one containing the piece at index 0
//		//	on itsPieceOrder) to the back of itsPieceOrder.
//		SendClumpToFrontOrBack(md, 0, SendClumpToBack);
//	}


unsigned int GetNum2DJigsawBackgroundTextureRepetitions(void)
{
	return NUM_2D_BACKGROUND_TEXTURE_REPETITIONS_JIGSAW;
}

void Get2DJigsawKleinAxisColors(
	float	someKleinAxisColors[2][4])	//	output;  premultiplied alpha
{
	someKleinAxisColors[0][0] = 1.00;
	someKleinAxisColors[0][1] = 0.00;
	someKleinAxisColors[0][2] = 0.00;
	someKleinAxisColors[0][3] = 1.00;

	someKleinAxisColors[1][0] = 0.00;
	someKleinAxisColors[1][1] = 0.00;
	someKleinAxisColors[1][2] = 1.00;
	someKleinAxisColors[1][3] = 1.00;
}


unsigned int GetNum2DJigsawSprites(
	unsigned int	aPuzzleSize)
{
	return aPuzzleSize * aPuzzleSize;	//	puzzle pieces
}

void Get2DJigsawSpritePlacementsOrTexturePlacements(
	ModelData		*md,							//	input
	unsigned int	aNumSprites,					//	input
	Placement2D		*aPlacementBuffer,				//	optional output;  big enough to hold aNumSprites Placement2D's
	double			(*aTexturePlacementBuffer)[4])	//	optional output;  array of ( u_min, v_min, u_max - u_min, v_max - v_min ) in bottom-up coordinates
{
	unsigned int	i,
					n;						//	puzzle size is n pieces by n pieces
	double			theNominalPieceSize;	//	1/n in n×n jigsaw puzzle
	JigsawPiece		*theJigsawPiece;
	double			theSideWest,
					theSideEast,
					theSideSouth,	//	bottom-up coordinates
					theSideNorth,	//	bottom-up coordinates
					theOffsetH,
					theOffsetV,
					theCenterH,
					theCenterV;
	Placement2D		theOptimizedPiecePlacement;

	GEOMETRY_GAMES_ASSERT(
		md->itsGame == Game2DJigsaw,
		"Game2DJigsaw must be active");

	n					= md->itsGameOf.Jigsaw2D.itsSize;
	theNominalPieceSize	= 1.0 / (double) n;
	
	GEOMETRY_GAMES_ASSERT(
		aNumSprites == GetNum2DJigsawSprites(n),
		"Internal error:  wrong buffer size");

	for (i = 0; i < n*n; i++)
	{
		//	Use the reverse of itsPieceOrder for bottom-to-top layering
		theJigsawPiece = md->itsGameOf.Jigsaw2D.itsPieceOrder[(n*n - 1) - i];

#ifdef MAKE_GAME_CHOICE_ICONS
#warning To create an n × n image, render the board at (3/2)n × (3/2)n	\
	then crop out the desired n × n square.  It'll be roughly			\
	at the upper left, but offset a few pixel from the edges.
		unsigned int	h = theJigsawPiece->itsLogicalH,
						v = theJigsawPiece->itsLogicalV;

		//	For the game choice icon, manually place the three pieces
		//	at logical position (h,v) = (0,0), (1,0) and (1,1).
		if (h == 0 && v == 0)
		{
			theJigsawPiece->itsPlacement.itsH	= -0.43;
			theJigsawPiece->itsPlacement.itsV	=  0.03;
			theNominalPieceSize	= 1.0 / (double) n;
		}
		else
		if (h == 1 && v == 0)
		{
			theJigsawPiece->itsPlacement.itsH	=  0.00;
			theJigsawPiece->itsPlacement.itsV	=  0.00;
			theNominalPieceSize	= 1.0 / (double) n;
		}
		else
		if (h == 1 && v == 1)
		{
			theJigsawPiece->itsPlacement.itsH	= -0.05;
			theJigsawPiece->itsPlacement.itsV	= +0.43;
			theNominalPieceSize	= 1.0 / (double) n;
		}
		else
		{
			//	Get all other pieces out of the way.

			theJigsawPiece->itsPlacement.itsH	= +0.375;
			theJigsawPiece->itsPlacement.itsV	= -0.375;
			theNominalPieceSize	= 0.01;
		}
#endif

		//	Choose an efficient bounding rectangle that encloses theJigsawPiece,
		//	taking into account that some sides have a tab, while other do not.
		theSideWest =
			theJigsawPiece->itsTabGenderWest == TabFemale ?
				-0.5 * theNominalPieceSize * MARGIN_FOR_BORDER :	//	narrow margin allowing for border outline
				-0.5 * theNominalPieceSize * MARGIN_FOR_TAB;		//	wide margin allowing for tab
		theSideEast =
			theJigsawPiece->itsTabGenderEast == TabFemale ?
				+0.5 * theNominalPieceSize * MARGIN_FOR_BORDER :	//	narrow margin allowing for border outline
				+0.5 * theNominalPieceSize * MARGIN_FOR_TAB;		//	wide margin allowing for tab
		theSideSouth =
			theJigsawPiece->itsTabGenderSouth == TabFemale ?
				-0.5 * theNominalPieceSize * MARGIN_FOR_BORDER :	//	narrow margin allowing for border outline
				-0.5 * theNominalPieceSize * MARGIN_FOR_TAB;		//	wide margin allowing for tab
		theSideNorth =
			theJigsawPiece->itsTabGenderNorth == TabFemale ?
				+0.5 * theNominalPieceSize * MARGIN_FOR_BORDER :	//	narrow margin allowing for border outline
				+0.5 * theNominalPieceSize * MARGIN_FOR_TAB;		//	wide margin allowing for tab

		//	Set the sprite's placement on the board.
		if (aPlacementBuffer != NULL)
		{
			theOffsetH = 0.5 * (theSideEast  + theSideWest );
			theOffsetV = 0.5 * (theSideNorth + theSideSouth);

			theOptimizedPiecePlacement.itsH		= theJigsawPiece->itsPlacement.itsH
												+ (theJigsawPiece->itsPlacement.itsFlip ? -theOffsetH : theOffsetH);
			theOptimizedPiecePlacement.itsV		= theJigsawPiece->itsPlacement.itsV + theOffsetV;
			theOptimizedPiecePlacement.itsFlip	= theJigsawPiece->itsPlacement.itsFlip;
			theOptimizedPiecePlacement.itsAngle	= 0.0;
			theOptimizedPiecePlacement.itsSizeH	= theSideEast  - theSideWest;
			theOptimizedPiecePlacement.itsSizeV	= theSideNorth - theSideSouth;

			aPlacementBuffer[i] = theOptimizedPiecePlacement;
		}

		//	Set the sprite's placement in the texture.
		if (aTexturePlacementBuffer != NULL)
		{
			//	The GPU function TorusGamesComputeFunctionMakeJigsawCollages()
			//	has spread the pieces out in a single large texture,
			//	conceptually with [0.0, 2.0] × [0.0, 2.0] coordinates.
			//	Note theJigsawPiece's center in the collage.
			//
			theCenterH = md->itsGameOf.Jigsaw2D.itsPuzzlePieceCollageFirstCenter
					   + theJigsawPiece->itsLogicalH * md->itsGameOf.Jigsaw2D.itsPuzzlePieceCollageStride;
			theCenterV = md->itsGameOf.Jigsaw2D.itsPuzzlePieceCollageFirstCenter
					   + theJigsawPiece->itsLogicalV * md->itsGameOf.Jigsaw2D.itsPuzzlePieceCollageStride;

			//	Each "texture placement" has the format
			//
			//		( u_min, v_min, u_max - u_min, v_max - v_min )
			//
			//	Even though we visualize collage in [0.0, 2.0] × [0.0, 2.0] coordinates,
			//	we must multiply by 0.5 at the last minute to convert
			//	to standard [0.0, 1.0] × [0.0, 1.0] texture coordinates.
			//
			aTexturePlacementBuffer[i][0] = 0.5 * (theCenterH + theSideWest);
			aTexturePlacementBuffer[i][1] = 0.5 * (theCenterV + theSideSouth);
			aTexturePlacementBuffer[i][2] = 0.5 * (theSideEast  - theSideWest);
			aTexturePlacementBuffer[i][3] = 0.5 * (theSideNorth - theSideSouth);
		}
	}
}
